﻿/// アーカイブボリュームの列挙を行うアーカイブ列挙処理クラス ArchiveEnumerator のヘッダファイル。

#if       !defined(INCLUDED_SZ_ARCHIVE_ENUMERATOR)
#define            INCLUDED_SZ_ARCHIVE_ENUMERATOR

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)
#include  <szForwardDeclarations.hpp>
#endif // !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)

SZ_NS_BEG(szpp)

class SZ_SPEC ArchiveEnumeratorImpl;

/// <summary>
/// アーカイブデータの格納ファイルの列挙を行うアーカイブ列挙処理クラス。
/// </summary>
/// <remarks>
/// epo では圧縮はもちろん展開するだけであっても必ずアーカイブの列挙を先に行うので、アーカイブ列挙部はアーカイブ展開部と分離した。
/// </remarks>
/// <notes>
/// 7-Zip ではファイル名（ヘッダ）も暗号化できるので、列挙するだけであってもパスワードを要求されうる。
/// さらにマルチボリュームの処理なども加えていくと、列挙と言ってもそれなりに複雑な処理を行うことになるので、Detector
/// や Extractor とは独立させておく方が適切と思われる。
/// </notes>
class SZ_SPEC ArchiveEnumerator
{
public:

  /// <summary>
  /// デフォルトコンストラクタ。
  /// </summary>
  //ArchiveEnumerator();

  /// <summary>
  /// デストラクタ。
  /// </summary>
  //~ArchiveEnumerator();

  /// <summary>
  /// アーカイブボリュームの列挙を行うメソッド。
  /// </summary>
  /// <details>
  /// <para>
  /// RAR タイプのマルチボリュームの場合は、個々のボリュームが一つのアーカイブとして列挙可能であるので、volume 引数で特定されるボリュームのみ列挙する。
  /// </para>
  /// <para>
  /// これに対して、7-Zip タイプのマルチボリュームの場合は、個々のボリュームが一つのアーカイブは不正であるので、
  /// クライアント側でボリュームを結合した仮想ストリームを作成するなどしてから、あたかも一つのボリュームであるかのようにして開くことになる。
  /// </para>
  /// </details>
  /// <param name="handler">列挙に用いるアーカイブハンドラのポインタ。ArchiveDetector の静的メソッドが返したポインタを与えるのが一般的。</param>
  /// <param name="volume">列挙するアーカイブボリューム。アーカイブボリュームを正常に列挙できた場合には内部でアーカイブプロパティなどが設定される。</param>
  /// <param name="searchLimitSize">自己展開ファイルなどの場合にアーカイブデータの開始位置を調べる領域のサイズの最大値。</param>
  /// <param name="openCallback">
  /// アーカイブを開く際の進行状況などを受け取るクライアントコールバックのポインタ。進行状況を表示する必要なければ NULL を与えてもよい。
  /// </param>
  /// <param name="pwdCallback">
  /// アーカイブを開く際にユーザーにパスワードの問い合わせを行うクライアントコールバックのポインタ。コールバックが必要なければ NULL
  /// を与えてもよいが、それではパスワード保護されたアーカイブを開けないという不都合がある。
  /// </param>
  /// <returns>
  /// 列挙が成功すればアーカイブの内容を表す ArchiveContent インスタンスのポインタを、失敗すれば NULL を返す。
  /// 拡張子によるハンドラ判定は誤りも起こりうるので、実際に Enumerate を呼び出して結果が得られることを確認するのが最も確実がアーカイブ形式判定方法と言える。
  /// </returns>
  /// <remarks>アーカイブストリームはメソッド内で利用し、すぐに解放される。</remarks>
  static ArchiveContent *Enumerate(
    const ArchiveHandler *handler,
    ArchiveVolume *volume,
    u64 searchLimitSize,
    ArchiveOpenClientCallback *openCallback,
    GetPasswordClientCallback *pwdCallback);

  /// <summary>
  /// ハンドラが不明なアーカイブボリュームの列挙を行うメソッド。
  /// </summary>
  /// <details>
  /// <para>
  /// RAR タイプのマルチボリュームの場合は、個々のボリュームが一つのアーカイブとして列挙可能であるので、volume 引数で特定されるボリュームのみ列挙する。
  /// </para>
  /// <para>
  /// これに対して、7-Zip タイプのマルチボリュームの場合は、個々のボリュームが一つのアーカイブは不正であるので、
  /// クライアント側でボリュームを結合した仮想ストリームを作成するなどしてから、あたかも一つのボリュームであるかのようにして開くことになる。
  /// </para>
  /// </details>
  /// <param name="handler">アーカイブを開くのに成功したハンドラのポインタを格納するためのアーカイブハンドラポインタのポインタ。ポインタを delete するのは呼び出し側の責任となる。</param>
  /// <param name="volume">列挙するアーカイブボリューム。アーカイブボリュームを正常に列挙できた場合には内部でアーカイブプロパティなどが設定される。</param>
  /// <param name="searchLimitSize">自己展開ファイルなどの場合にアーカイブデータの開始位置を調べる領域のサイズの最大値。</param>
  /// <param name="openCallback">
  /// アーカイブを開く際の進行状況などを受け取るクライアントコールバックのポインタ。進行状況を表示する必要なければ NULL を与えてもよい。
  /// </param>
  /// <param name="pwdCallback">
  /// アーカイブを開く際にユーザーにパスワードの問い合わせを行うクライアントコールバックのポインタ。コールバックが必要なければ NULL
  /// を与えてもよいが、それではパスワード保護されたアーカイブを開けないという不都合がある。
  /// </param>
  /// <returns>
  /// 列挙が成功すればアーカイブの内容を表す ArchiveContent インスタンスのポインタを、失敗すれば NULL を返す。
  /// </returns>
  /// <remarks>アーカイブストリームはメソッド内で利用し、すぐに解放される。</remarks>
  static ArchiveContent *EnumerateUnknown(
    ArchiveHandler **handler,
    ArchiveVolume *volume,
    u64 searchLimitSize,
    ArchiveOpenClientCallback *openCallback,
    GetPasswordClientCallback *pwdCallback);

};

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_ARCHIVE_ENUMERATOR)
